<?php

namespace Database\Seeders;

use App\Models\Client;
use App\Models\Order;
use App\Models\Rating;
use App\Models\Service;
use App\Models\Technician;
use Illuminate\Database\Seeder;
use Illuminate\Support\Carbon;

class OrderSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $clients = Client::where('is_active', true)->get();
        $services = Service::where('is_active', true)->get();
        $technicians = Technician::where('is_active', true)->get();

        if ($clients->isEmpty() || $services->isEmpty()) {
            $this->command->warn('لا توجد عملاء أو خدمات متاحة. يرجى إنشاء بيانات تجريبية أولاً.');
            return;
        }

        $statuses = ['pending', 'confirmed', 'in_progress', 'completed', 'cancelled'];
        $types = ['normal', 'urgent'];
        
        $orders = [];

        // إنشاء 50 طلب
        for ($i = 0; $i < 50; $i++) {
            $client = $clients->random();
            $service = $services->random();
            $technician = $technicians->isNotEmpty() ? $technicians->random() : null;
            $status = $statuses[array_rand($statuses)];
            $type = $types[array_rand($types)];
            
            // تحديد التواريخ بناءً على الحالة
            $scheduledDate = null;
            $completedDate = null;
            
            if ($status !== 'pending') {
                $scheduledDate = Carbon::now()->subDays(rand(1, 30))->addHours(rand(1, 23));
            }
            
            if ($status === 'completed') {
                $completedDate = $scheduledDate ? $scheduledDate->copy()->addHours(rand(2, 8)) : Carbon::now()->subDays(rand(1, 15));
            }

            // حساب الأسعار
            $price = $service->price ?? rand(100, 1000);
            $fee = round($price * 0.03, 2); // 3% رسوم
            $discount = rand(0, 1) ? rand(0, 50) : 0; // خصم عشوائي
            $total = $price + $fee - $discount;

            // سيتم توليد order_number تلقائياً من Order model بناءً على الصيغة في الإعدادات
            $order = Order::create([
                'client_id' => $client->id,
                'service_id' => $service->id,
                'technician_id' => $technician?->id,
                'type' => $type,
                'status' => $status,
                'scheduled_date' => $scheduledDate,
                'completed_date' => $completedDate,
                'price' => $price,
                'fee' => $fee,
                'discount' => $discount,
                'total' => $total,
                'address' => $client->address ?? 'عنوان تجريبي',
                'city' => $client->city ?? 'الرياض',
                'notes' => rand(0, 1) ? 'ملاحظات إدارية تجريبية' : null,
                'client_notes' => rand(0, 1) ? 'ملاحظات العميل' : null,
                'created_at' => Carbon::now()->subDays(rand(1, 60)),
                'updated_at' => Carbon::now()->subDays(rand(0, 30)),
            ]);

            $orders[] = $order;

            // إنشاء تقييمات للطلبات المكتملة (70% من الطلبات المكتملة)
            if ($status === 'completed' && rand(1, 10) <= 7) {
                Rating::create([
                    'order_id' => $order->id,
                    'client_id' => $client->id,
                    'rating' => rand(3, 5), // تقييمات جيدة بشكل عام
                    'comment' => $this->getRandomComment(),
                    'created_at' => $completedDate ? $completedDate->copy()->addHours(rand(1, 24)) : Carbon::now()->subDays(rand(1, 10)),
                ]);
            }
        }

        $this->command->info('تم إنشاء ' . count($orders) . ' طلب بنجاح!');
        $this->command->info('تم إنشاء ' . Rating::count() . ' تقييم بنجاح!');
    }

    private function getRandomComment(): string
    {
        $comments = [
            'خدمة ممتازة، الفني كان محترف جداً',
            'راضٍ جداً عن الخدمة، سأطلب مرة أخرى',
            'خدمة جيدة ولكن يمكن تحسينها',
            'الفني وصل في الوقت المحدد، شكراً لكم',
            'الخدمة كانت سريعة وفعالة',
            'جودة العمل ممتازة، أنصح بالتعامل معهم',
            'خدمة جيدة ولكن السعر مرتفع قليلاً',
            'الفني كان مهذب ومحترف',
            'الخدمة تمت بشكل سريع وممتاز',
            'راضٍ عن الخدمة بشكل عام',
            'خدمة ممتازة، شكراً لكم',
            'الفني كان محترف والخدمة ممتازة',
        ];

        return $comments[array_rand($comments)];
    }
}
