<?php

namespace App\Filament\Technician\Resources;

use App\Filament\Technician\Resources\TechnicianOrderResource\Pages;
use App\Models\Order;
use Filament\Forms;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

class TechnicianOrderResource extends Resource
{
    protected static ?string $model = Order::class;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-clipboard-document-list';

    protected static ?string $navigationLabel = 'طلباتي';

    protected static ?string $modelLabel = 'طلب';

    protected static ?string $pluralModelLabel = 'طلباتي';

    protected static ?int $navigationSort = 1;

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->where('technician_id', Auth::guard('technician')->id());
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('order_number')
                    ->label('رقم الطلب')
                    ->searchable()
                    ->sortable()
                    ->copyable(),

                Tables\Columns\TextColumn::make('client.name')
                    ->label('العميل')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('service.name')
                    ->label('الخدمة')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('type')
                    ->label('النوع')
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'urgent' => 'مستعجل',
                        'normal' => 'عادي',
                        default => $state,
                    })
                    ->color(fn (string $state): string => match($state) {
                        'urgent' => 'danger',
                        'normal' => 'info',
                        default => 'gray',
                    }),

                Tables\Columns\TextColumn::make('status')
                    ->label('الحالة')
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'pending' => 'قيد الانتظار',
                        'confirmed' => 'مؤكد',
                        'in_progress' => 'قيد التنفيذ',
                        'completed' => 'مكتمل',
                        'cancelled' => 'ملغي',
                        default => $state,
                    })
                    ->color(fn (string $state): string => match($state) {
                        'pending' => 'warning',
                        'confirmed' => 'info',
                        'in_progress' => 'primary',
                        'completed' => 'success',
                        'cancelled' => 'danger',
                        default => 'gray',
                    }),

                Tables\Columns\TextColumn::make('scheduled_date')
                    ->label('تاريخ الموعد')
                    ->dateTime('Y-m-d H:i')
                    ->sortable(),

                Tables\Columns\TextColumn::make('technician_commission')
                    ->label('مستحقي')
                    ->money('SAR')
                    ->sortable()
                    ->color('success')
                    ->weight('bold'),

                Tables\Columns\TextColumn::make('address')
                    ->label('العنوان')
                    ->limit(30)
                    ->toggleable(isToggledHiddenByDefault: true),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->label('الحالة')
                    ->options([
                        'pending' => 'قيد الانتظار',
                        'confirmed' => 'مؤكد',
                        'in_progress' => 'قيد التنفيذ',
                        'completed' => 'مكتمل',
                        'cancelled' => 'ملغي',
                    ]),

                Tables\Filters\SelectFilter::make('type')
                    ->label('النوع')
                    ->options([
                        'normal' => 'عادي',
                        'urgent' => 'مستعجل',
                    ]),
            ])
            ->actions([
                \Filament\Actions\ViewAction::make(),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function infolist(\Filament\Schemas\Schema $schema): \Filament\Schemas\Schema
    {
        return $schema
            ->schema([
                \Filament\Schemas\Components\Section::make('معلومات الطلب')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('order_number')
                            ->label('رقم الطلب')
                            ->copyable(),
                        \Filament\Infolists\Components\TextEntry::make('client.name')
                            ->label('العميل'),
                        \Filament\Infolists\Components\TextEntry::make('service.name')
                            ->label('الخدمة'),
                        \Filament\Infolists\Components\TextEntry::make('type')
                            ->label('النوع')
                            ->badge()
                            ->formatStateUsing(fn (string $state): string => match($state) {
                                'urgent' => 'مستعجل',
                                'normal' => 'عادي',
                                default => $state,
                            })
                            ->color(fn (string $state): string => match($state) {
                                'urgent' => 'danger',
                                'normal' => 'info',
                                default => 'gray',
                            }),
                        \Filament\Infolists\Components\TextEntry::make('status')
                            ->label('الحالة')
                            ->badge()
                            ->formatStateUsing(fn (string $state): string => match($state) {
                                'pending' => 'قيد الانتظار',
                                'confirmed' => 'مؤكد',
                                'in_progress' => 'قيد التنفيذ',
                                'completed' => 'مكتمل',
                                'cancelled' => 'ملغي',
                                default => $state,
                            })
                            ->color(fn (string $state): string => match($state) {
                                'pending' => 'warning',
                                'confirmed' => 'info',
                                'in_progress' => 'primary',
                                'completed' => 'success',
                                'cancelled' => 'danger',
                                default => 'gray',
                            }),
                        \Filament\Infolists\Components\TextEntry::make('scheduled_date')
                            ->label('تاريخ الموعد')
                            ->formatStateUsing(fn ($state) => $state ? \Carbon\Carbon::parse($state)->format('Y-m-d H:i') : '-'),
                        \Filament\Infolists\Components\TextEntry::make('completed_date')
                            ->label('تاريخ الإنجاز')
                            ->formatStateUsing(fn ($state) => $state ? \Carbon\Carbon::parse($state)->format('Y-m-d H:i') : '-'),
                    ])
                    ->columns(3),

                \Filament\Schemas\Components\Section::make('المعلومات المالية')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('total')
                            ->label('إجمالي الطلب')
                            ->money('SAR')
                            ->icon('heroicon-o-currency-dollar'),
                        \Filament\Infolists\Components\TextEntry::make('technician_commission')
                            ->label('مستحقي من هذا الطلب')
                            ->money('SAR')
                            ->color('success')
                            ->weight('bold')
                            ->size('lg')
                            ->icon('heroicon-o-banknotes'),
                    ])
                    ->columns(2),

                \Filament\Schemas\Components\Section::make('العنوان')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('address')
                            ->label('العنوان')
                            ->default('-'),
                        \Filament\Infolists\Components\TextEntry::make('city')
                            ->label('المدينة')
                            ->default('-'),
                    ])
                    ->columns(2),

                \Filament\Schemas\Components\Section::make('ملاحظات')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('notes')
                            ->label('ملاحظات إدارية')
                            ->default('لا توجد ملاحظات')
                            ->columnSpanFull(),
                        \Filament\Infolists\Components\TextEntry::make('client_notes')
                            ->label('ملاحظات العميل')
                            ->default('لا توجد ملاحظات')
                            ->columnSpanFull(),
                    ]),

                \Filament\Schemas\Components\Section::make('التقييم')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('rating.rating')
                            ->label('التقييم')
                            ->formatStateUsing(fn ($state) => $state ? str_repeat('⭐', $state) . str_repeat('☆', 5 - $state) : 'لم يتم التقييم'),
                        \Filament\Infolists\Components\TextEntry::make('rating.comment')
                            ->label('تعليق التقييم')
                            ->default('لا يوجد تعليق'),
                    ])
                    ->columns(2),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTechnicianOrders::route('/'),
            'view' => Pages\ViewTechnicianOrder::route('/{record}'),
        ];
    }

    public static function canCreate(): bool
    {
        return false;
    }

    public static function canEdit($record): bool
    {
        return false;
    }

    public static function canDelete($record): bool
    {
        return false;
    }
}

