<?php

namespace App\Filament\Technician\Pages;

use Filament\Actions\Action;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Actions;
use Filament\Schemas\Components\EmbeddedSchema;
use Filament\Schemas\Components\Form;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Concerns\InteractsWithSchemas;
use Filament\Schemas\Contracts\HasSchemas;
use Filament\Schemas\Schema;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class TechnicianProfile extends Page implements HasSchemas
{
    use InteractsWithSchemas;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-user-circle';

    protected static ?string $navigationLabel = 'الملف الشخصي';

    protected static ?string $title = 'الملف الشخصي';

    protected static ?int $navigationSort = 3;

    public ?string $name = '';
    public ?string $email = '';
    public ?string $phone = '';
    public ?string $service_name = '';
    public ?string $bio = '';
    public ?string $current_password = '';
    public ?string $new_password = '';
    public ?string $new_password_confirmation = '';

    public function mount(): void
    {
        $technician = Auth::guard('technician')->user();

        $this->name = $technician->name;
        $this->email = $technician->email;
        $this->phone = $technician->phone;
        $this->service_name = $technician->service?->getTranslation('name', 'ar') ?? $technician->service?->name ?? 'غير محدد';
        $this->bio = $technician->bio;
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('المعلومات الشخصية')
                    ->icon('heroicon-o-user')
                    ->description('بيانات الحساب الأساسية')
                    ->components([
                        TextInput::make('name')
                            ->label('الاسم')
                            ->required(),
                        TextInput::make('email')
                            ->label('البريد الإلكتروني')
                            ->email()
                            ->required(),
                        TextInput::make('phone')
                            ->label('رقم الهاتف')
                            ->tel(),
                        TextInput::make('service_name')
                            ->label('نوع الخدمة')
                            ->disabled()
                            ->helperText('لتغيير نوع الخدمة تواصل مع الإدارة'),
                        Textarea::make('bio')
                            ->label('نبذة عنك')
                            ->rows(3)
                            ->columnSpanFull(),
                    ])
                    ->columns(2),

                Section::make('تغيير كلمة المرور')
                    ->icon('heroicon-o-lock-closed')
                    ->description('اتركها فارغة إذا لم ترد تغيير كلمة المرور')
                    ->components([
                        TextInput::make('current_password')
                            ->label('كلمة المرور الحالية')
                            ->password()
                            ->revealable(),
                        TextInput::make('new_password')
                            ->label('كلمة المرور الجديدة')
                            ->password()
                            ->revealable()
                            ->minLength(8),
                        TextInput::make('new_password_confirmation')
                            ->label('تأكيد كلمة المرور الجديدة')
                            ->password()
                            ->revealable()
                            ->same('new_password'),
                    ])
                    ->columns(3),

                Section::make('المستندات')
                    ->icon('heroicon-o-document-duplicate')
                    ->description('رفع المستندات الخاصة بك (PDF, Excel, Images)')
                    ->components([
                        SpatieMediaLibraryFileUpload::make('documents')
                            ->label('المستندات')
                            ->collection('documents')
                            ->acceptedFileTypes(['application/pdf', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'image/jpeg', 'image/png', 'image/jpg'])
                            ->multiple()
                            ->maxFiles(10)
                            ->maxSize(10240) // 10MB
                            ->downloadable()
                            ->previewable()
                            ->openable()
                            ->helperText('يمكنك رفع ملفات PDF و Excel و الصور. الحد الأقصى 10 ملفات، حجم كل ملف لا يتجاوز 10 ميجابايت')
                            ->columnSpanFull(),
                    ]),
            ]);
    }

    public function content(Schema $schema): Schema
    {
        return $schema
            ->components([
                Form::make([EmbeddedSchema::make('form')])
                    ->id('form')
                    ->livewireSubmitHandler('save')
                    ->footer([
                        Actions::make($this->getFormActions())
                            ->alignment('end')
                            ->key('form-actions'),
                    ]),
            ]);
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')
                ->label('حفظ التغييرات')
                ->submit('save')
                ->icon('heroicon-o-check')
                ->color('success'),
        ];
    }

    public function save(): void
    {
        $technician = Auth::guard('technician')->user();

        $technician->name = $this->name;
        $technician->email = $this->email;
        $technician->phone = $this->phone;
        $technician->bio = $this->bio;

        if (!empty($this->current_password) && !empty($this->new_password)) {
            if (!Hash::check($this->current_password, $technician->password)) {
                Notification::make()
                    ->title('كلمة المرور الحالية غير صحيحة')
                    ->danger()
                    ->send();
                return;
            }

            $technician->password = Hash::make($this->new_password);
        }

        $technician->save();

        $this->current_password = '';
        $this->new_password = '';
        $this->new_password_confirmation = '';

        Notification::make()
            ->title('تم تحديث الملف الشخصي بنجاح')
            ->success()
            ->send();
    }
}
