<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TechnicianResource\Pages;
use App\Models\Technician;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables\Table;

class TechnicianResource extends Resource
{
    protected static ?string $model = Technician::class;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-user-group';

    protected static ?string $navigationLabel = 'الفنيون';

    protected static ?string $modelLabel = 'فني';

    protected static ?string $pluralModelLabel = 'الفنيون';

    protected static string | \UnitEnum | null $navigationGroup = 'إدارة الخدمات';

    protected static ?int $navigationSort = 2;

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('الصورة الشخصية')
                    ->components([
                        SpatieMediaLibraryFileUpload::make('profile_photo')
                            ->label('صورة الفني')
                            ->collection('profile_photo')
                            ->image()
                            ->avatar()
                            ->maxSize(2048)
                            ->columnSpanFull(),
                    ]),
                Section::make('البيانات الشخصية')
                    ->components([
                        TextInput::make('name')
                            ->label('الاسم الكامل')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('مثال: ياسين علي'),
                        TextInput::make('email')
                            ->label('البريد الإلكتروني')
                            ->email()
                            ->required()
                            ->maxLength(255)
                            ->unique(ignoreRecord: true)
                            ->placeholder('example@gmail.com'),
                        TextInput::make('phone')
                            ->label('رقم الهاتف')
                            ->tel()
                            ->maxLength(255)
                            ->placeholder('+966 5XX XXX XXXX'),
                        TextInput::make('password')
                            ->label('كلمة المرور')
                            ->password()
                            ->revealable()
                            ->required(fn (string $operation): bool => $operation === 'create')
                            ->dehydrated(fn (?string $state) => filled($state))
                            ->minLength(8)
                            ->helperText('كلمة المرور لدخول لوحة تحكم الفني')
                            ->placeholder('أدخل كلمة المرور'),
                        Select::make('gender')
                            ->label('الجنس')
                            ->options([
                                'Male' => 'ذكر',
                                'Female' => 'أنثى',
                            ])
                            ->native(false),
                    ])->columns(2),
                Section::make('معلومات الخدمة')
                    ->components([
                        Select::make('service_id')
                            ->label('نوع الخدمة')
                            ->relationship('service', 'name')
                            ->searchable()
                            ->preload()
                            ->required()
                            ->helperText('اختر نوع الخدمة التي يقدمها الفني'),
                        TextInput::make('hourly_rate')
                            ->label('سعر الساعة')
                            ->numeric()
                            ->prefix('ر.س')
                            ->step(0.01)
                            ->placeholder('0.00'),
                        Checkbox::make('is_active')
                            ->label('نشط')
                            ->default(true)
                            ->helperText('هل الفني متاح حالياً لاستقبال الطلبات؟'),
                        Textarea::make('bio')
                            ->label('نبذة عن الفني')
                            ->rows(3)
                            ->columnSpanFull()
                            ->placeholder('وصف مختصر عن خبرات الفني...'),
                    ])->columns(2),
                Section::make('المستندات')
                    ->icon('heroicon-o-document-duplicate')
                    ->description('رفع المستندات الخاصة بالفني (PDF, Excel, Images)')
                    ->components([
                        SpatieMediaLibraryFileUpload::make('documents')
                            ->label('المستندات')
                            ->collection('documents')
                            ->acceptedFileTypes(['application/pdf', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'image/jpeg', 'image/png', 'image/jpg'])
                            ->multiple()
                            ->maxFiles(10)
                            ->maxSize(10240) // 10MB
                            ->downloadable()
                            ->previewable()
                            ->openable()
                            ->helperText('يمكنك رفع ملفات PDF و Excel و الصور. الحد الأقصى 10 ملفات، حجم كل ملف لا يتجاوز 10 ميجابايت')
                            ->columnSpanFull(),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                SpatieMediaLibraryImageColumn::make('profile_photo')
                    ->label('الصورة')
                    ->collection('profile_photo')
                    ->circular()
                    ->defaultImageUrl(fn () => 'https://ui-avatars.com/api/?name=T&background=f59e0b&color=fff'),
                Tables\Columns\TextColumn::make('name')
                    ->label('الاسم الكامل')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),
                Tables\Columns\TextColumn::make('email')
                    ->label('البريد الإلكتروني')
                    ->searchable()
                    ->sortable()
                    ->icon('heroicon-o-envelope'),
                Tables\Columns\TextColumn::make('phone')
                    ->label('رقم الهاتف')
                    ->searchable()
                    ->icon('heroicon-o-phone'),
                Tables\Columns\TextColumn::make('gender')
                    ->label('الجنس')
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'Male' => 'ذكر',
                        'Female' => 'أنثى',
                        default => $state,
                    })
                    ->color(fn (string $state): string => match ($state) {
                        'Male' => 'info',
                        'Female' => 'success',
                        default => 'gray',
                    }),
                Tables\Columns\TextColumn::make('service.name')
                    ->label('نوع الخدمة')
                    ->badge()
                    ->color('warning')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('hourly_rate')
                    ->label('سعر الساعة')
                    ->money('SAR')
                    ->sortable(),
                Tables\Columns\IconColumn::make('is_active')
                    ->label('نشط')
                    ->boolean(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('تاريخ الإضافة')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('service_id')
                    ->label('نوع الخدمة')
                    ->relationship('service', 'name')
                    ->searchable()
                    ->preload(),
                Tables\Filters\SelectFilter::make('gender')
                    ->label('الجنس')
                    ->options([
                        'Male' => 'ذكر',
                        'Female' => 'أنثى',
                    ]),
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->placeholder('الكل')
                    ->trueLabel('نشط فقط')
                    ->falseLabel('غير نشط فقط'),
            ])
            ->actions([
                \Filament\Actions\Action::make('impersonate')
                    ->label('دخول كفني')
                    ->icon('heroicon-o-finger-print')
                    ->color('warning')
                    ->requiresConfirmation()
                    ->modalHeading('تسجيل الدخول كفني')
                    ->modalDescription(fn (Technician $record) => "سيتم تسجيل دخولك كـ \"{$record->name}\" في لوحة تحكم الفني")
                    ->modalSubmitActionLabel('تأكيد الدخول')
                    ->action(function (Technician $record) {
                        session()->put('impersonating_from_admin', auth()->id());
                        auth()->guard('technician')->login($record);
                        redirect('/technician');
                    }),
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                DeleteBulkAction::make(),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTechnicians::route('/'),
            'create' => Pages\CreateTechnician::route('/create'),
            'edit' => Pages\EditTechnician::route('/{record}/edit'),
        ];
    }
}

