<?php

namespace App\Filament\Resources;

use App\Exports\ServicesExport;
use App\Filament\Resources\ServiceResource\Pages;
use App\Models\Category;
use App\Models\Service;
use Barryvdh\DomPDF\Facade\Pdf;
use Filament\Actions\BulkAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Maatwebsite\Excel\Facades\Excel;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;

class ServiceResource extends Resource
{
    protected static ?string $model = Service::class;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-wrench-screwdriver';

    protected static ?string $navigationLabel = 'الخدمات';

    protected static ?string $modelLabel = 'خدمة';

    protected static ?string $pluralModelLabel = 'الخدمات';

    protected static string | \UnitEnum | null $navigationGroup = 'إدارة الخدمات';

    protected static ?int $navigationSort = 1;

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Grid::make()
                    ->schema([
                        // ========== العمود الأيسر - المحتوى الرئيسي ==========
                        Section::make('معلومات الخدمة')
                            ->description('أدخل بيانات الخدمة بالعربية والإنجليزية - جميع الحقول مطلوبة')
                            ->icon('heroicon-o-language')
                            ->iconColor('primary')
                            ->collapsible()
                            ->collapsed(false)
                            ->components([
                                Tabs::make('Translations')
                                    ->tabs([
                                        Tab::make('🇸🇦 العربية')
                                            ->icon('heroicon-o-language')
                                            ->badge(fn ($get) => $get('name.ar') ? '✓' : null)
                                            ->badgeColor('success')
                                            ->schema([
                                                TextInput::make('name.ar')
                                                    ->label('اسم الخدمة')
                                                    ->required()
                                                    ->maxLength(255)
                                                    ->placeholder('مثال: تمديد أسلاك كهربائية')
                                                    ->prefixIcon('heroicon-o-document-text')
                                                    ->suffixIcon('heroicon-o-pencil')
                                                    ->helperText('اسم الخدمة باللغة العربية - سيظهر للعملاء')
                                                    ->columnSpanFull(),
                                                Textarea::make('description.ar')
                                                    ->label('وصف الخدمة')
                                                    ->rows(6)
                                                    ->placeholder('أدخل وصف تفصيلي للخدمة...')
                                                    ->helperText('وصف تفصيلي للخدمة بالعربية - يظهر في صفحة الخدمة')
                                                    ->columnSpanFull(),
                                            ]),
                                        Tab::make('🇬🇧 English')
                                            ->icon('heroicon-o-language')
                                            ->badge(fn ($get) => $get('name.en') ? '✓' : null)
                                            ->badgeColor('success')
                                            ->schema([
                                                TextInput::make('name.en')
                                                    ->label('Service Name')
                                                    ->required()
                                                    ->maxLength(255)
                                                    ->placeholder('e.g. Electrical Wiring')
                                                    ->prefixIcon('heroicon-o-document-text')
                                                    ->suffixIcon('heroicon-o-pencil')
                                                    ->helperText('Service name in English - will be displayed to clients')
                                                    ->columnSpanFull(),
                                                Textarea::make('description.en')
                                                    ->label('Description')
                                                    ->rows(6)
                                                    ->placeholder('Enter a detailed description...')
                                                    ->helperText('Detailed service description in English - appears on service page')
                                                    ->columnSpanFull(),
                                            ]),
                                    ])
                                    ->columnSpanFull(),
                            ])
                            ->columnSpan([
                                'default' => 1,
                                'sm' => 1,
                                'md' => 1,
                                'lg' => 5,
                                'xl' => 5,
                            ]),

                        // ========== العمود الأيمن - الإعدادات (جنب بعض) ==========
                        Section::make('القسم')
                            ->description('اختر القسم المناسب للخدمة')
                            ->icon('heroicon-o-squares-2x2')
                            ->iconColor('info')
                            ->collapsible()
                            ->collapsed(false)
                            ->components([
                                Select::make('category_id')
                                    ->label('اختر القسم')
                                    ->relationship('category', 'name')
                                    ->getOptionLabelFromRecordUsing(fn ($record) => $record->getTranslation('name', 'ar') . ' / ' . $record->getTranslation('name', 'en'))
                                    ->required()
                                    ->searchable()
                                    ->preload()
                                    ->prefixIcon('heroicon-o-folder')
                                    ->columnSpanFull(),
                            ])
                            ->columnSpan([
                                'default' => 1,
                                'sm' => 1,
                                'md' => 1,
                                'lg' => 2,
                                'xl' => 2,
                            ]),

                        Section::make('المظهر')
                            ->description('اختر الأيقونة المناسبة')
                            ->icon('heroicon-o-paint-brush')
                            ->iconColor('warning')
                            ->collapsible()
                            ->collapsed(false)
                            ->components([
                                Select::make('icon')
                                    ->label('الأيقونة')
                                    ->options([
                                        'heroicon-o-wrench' => '🔧 مفتاح ربط',
                                        'heroicon-o-wrench-screwdriver' => '🛠️ أدوات',
                                        'heroicon-o-bolt' => '⚡ كهرباء',
                                        'heroicon-o-light-bulb' => '💡 إضاءة',
                                        'heroicon-o-fire' => '🔥 تدفئة',
                                        'heroicon-o-beaker' => '💧 سباكة',
                                        'heroicon-o-paint-brush' => '🎨 دهان',
                                        'heroicon-o-cube' => '📦 نجارة',
                                        'heroicon-o-cog-6-tooth' => '⚙️ تكييف',
                                        'heroicon-o-cpu-chip' => '🔌 إلكترونيات',
                                        'heroicon-o-tv' => '📺 تلفزيون',
                                        'heroicon-o-computer-desktop' => '💻 كمبيوتر',
                                        'heroicon-o-device-phone-mobile' => '📱 جوال',
                                        'heroicon-o-sparkles' => '✨ تنظيف',
                                        'heroicon-o-shield-check' => '🛡️ فحص',
                                        'heroicon-o-key' => '🔑 أقفال',
                                        'heroicon-o-home-modern' => '🏠 منزل',
                                        'heroicon-o-building-office' => '🏢 مبنى',
                                        'heroicon-o-scissors' => '✂️ حدائق',
                                        'heroicon-o-camera' => '📷 كاميرات',
                                    ])
                                    ->searchable()
                                    ->native(false)
                                    ->placeholder('اختر أيقونة')
                                    ->prefixIcon('heroicon-o-photo')
                                    ->columnSpanFull(),
                            ])
                            ->columnSpan([
                                'default' => 1,
                                'sm' => 1,
                                'md' => 1,
                                'lg' => 2,
                                'xl' => 2,
                            ]),

                        Section::make('الحالة والترتيب')
                            ->description('إعدادات عرض الخدمة')
                            ->icon('heroicon-o-cog-6-tooth')
                            ->iconColor('success')
                            ->collapsible()
                            ->collapsed(false)
                            ->components([
                                Toggle::make('is_active')
                                    ->label('الخدمة نشطة')
                                    ->default(true)
                                    ->onColor('success')
                                    ->offColor('danger')
                                    ->onIcon('heroicon-m-check')
                                    ->offIcon('heroicon-m-x-mark')
                                    ->inline(false)
                                    ->helperText('تفعيل/تعطيل ظهور الخدمة في النظام')
                                    ->columnSpanFull(),
                                TextInput::make('sort_order')
                                    ->label('ترتيب العرض')
                                    ->numeric()
                                    ->default(0)
                                    ->minValue(0)
                                    ->placeholder('0')
                                    ->prefixIcon('heroicon-o-arrows-up-down')
                                    ->suffixIcon('heroicon-o-list-bullet')
                                    ->helperText('رقم أقل = يظهر أولاً في القائمة')
                                    ->columnSpanFull(),
                            ])
                            ->columnSpan([
                                'default' => 1,
                                'sm' => 1,
                                'md' => 1,
                                'lg' => 3,
                                'xl' => 3,
                            ]),
                    ]),

                // ========== قسم التسعير منفصل في الأسفل ==========
                Section::make('التسعير والنقاط')
                    ->description('حدد سعر الخدمة والنقاط المكتسبة للعملاء')
                    ->icon('heroicon-o-banknotes')
                    ->iconColor('warning')
                    ->collapsible()
                    ->collapsed(false)
                    ->components([
                        Grid::make([
                            'default' => 1,
                            'sm' => 1,
                            'md' => 2,
                            'lg' => 2,
                            'xl' => 2,
                        ])
                            ->schema([
                                TextInput::make('price')
                                    ->label('السعر')
                                    ->numeric()
                                    ->prefix('ر.س')
                                    ->step(0.01)
                                    ->minValue(0)
                                    ->placeholder('0.00')
                                    ->prefixIcon('heroicon-o-currency-dollar')
                                    ->suffixIcon('heroicon-o-banknotes')
                                    ->helperText('سعر الخدمة بالريال السعودي')
                                    ->required(),
                                TextInput::make('points')
                                    ->label('النقاط المكتسبة')
                                    ->numeric()
                                    ->default(0)
                                    ->minValue(0)
                                    ->suffix('نقطة')
                                    ->prefixIcon('heroicon-o-star')
                                    ->suffixIcon('heroicon-o-gift')
                                    ->helperText('عدد النقاط التي يحصل عليها العميل عند شراء هذه الخدمة')
                                    ->columnSpan(2),
                            ]),
                    ])
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\IconColumn::make('icon')
                    ->label('الأيقونة')
                    ->icon(fn (string $state): string => $state ?: 'heroicon-o-wrench-screwdriver')
                    ->color('primary'),
                Tables\Columns\TextColumn::make('category.name')
                    ->label('القسم')
                    ->formatStateUsing(fn ($record) => $record->category ? ($record->category->getTranslation('name', 'ar') . ' / ' . $record->category->getTranslation('name', 'en')) : '')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('name')
                    ->label('اسم الخدمة')
                    ->formatStateUsing(fn ($record) => $record->getTranslation('name', 'ar') . ' / ' . $record->getTranslation('name', 'en'))
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('price')
                    ->label('السعر')
                    ->money('SAR')
                    ->sortable(),
                Tables\Columns\TextColumn::make('points')
                    ->label('النقاط المكتسبة')
                    ->suffix(' نقطة')
                    ->badge()
                    ->color('success')
                    ->sortable(),
                Tables\Columns\IconColumn::make('is_active')
                    ->label('نشط')
                    ->boolean(),
                Tables\Columns\TextColumn::make('sort_order')
                    ->label('الترتيب')
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('category_id')
                    ->label('القسم')
                    ->relationship('category', 'name')
                    ->getOptionLabelFromRecordUsing(fn ($record) => $record->getTranslation('name', 'ar') . ' / ' . $record->getTranslation('name', 'en'))
                    ->searchable()
                    ->preload(),
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('نشط')
                    ->placeholder('الكل')
                    ->trueLabel('نشط فقط')
                    ->falseLabel('غير نشط فقط'),
            ])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                DeleteBulkAction::make(),
                BulkAction::make('export_excel')
                    ->label('تصدير Excel')
                    ->icon('heroicon-o-arrow-down-tray')
                    ->color('success')
                    ->action(function ($records) {
                        $services = $records->load('category');
                        return Excel::download(new ServicesExport($services), 'services-' . now()->format('Y-m-d') . '.xlsx');
                    }),
                BulkAction::make('export_pdf')
                    ->label('تصدير PDF')
                    ->icon('heroicon-o-document-arrow-down')
                    ->color('danger')
                    ->action(function ($records) {
                        $services = $records->load('category');
                        $pdf = Pdf::loadView('exports.services-pdf', ['services' => $services])
                            ->setPaper('a4', 'landscape')
                            ->setOption('isRemoteEnabled', true);
                        return response()->streamDownload(function () use ($pdf) {
                            echo $pdf->output();
                        }, 'services-' . now()->format('Y-m-d') . '.pdf');
                    }),
            ])
            ->defaultSort('sort_order');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListServices::route('/'),
            'create' => Pages\CreateService::route('/create'),
            'edit' => Pages\EditService::route('/{record}/edit'),
        ];
    }
}
