<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PointTransactionResource\Pages;
use App\Models\PointTransaction;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;

class PointTransactionResource extends Resource
{
    protected static ?string $model = PointTransaction::class;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-sparkles';

    protected static ?string $navigationLabel = 'معاملات النقاط';

    protected static ?string $modelLabel = 'معاملة نقاط';

    protected static ?string $pluralModelLabel = 'معاملات النقاط';

    protected static string | \UnitEnum | null $navigationGroup = 'إدارة النقاط';

    protected static ?int $navigationSort = 1;

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('معلومات المعاملة')
                    ->components([
                        Select::make('client_id')
                            ->label('العميل')
                            ->relationship('client', 'name')
                            ->required()
                            ->searchable()
                            ->preload(),
                        Select::make('type')
                            ->label('نوع المعاملة')
                            ->options([
                                'earned' => 'اكتساب نقاط',
                                'spent' => 'استخدام نقاط',
                                'expired' => 'انتهاء نقاط',
                                'admin_adjustment' => 'تعديل إداري',
                            ])
                            ->required()
                            ->default('earned')
                            ->native(false)
                            ->reactive(),
                        TextInput::make('points')
                            ->label('عدد النقاط')
                            ->numeric()
                            ->required()
                            ->default(0)
                            ->suffix('نقطة')
                            ->helperText('عدد النقاط (موجب للإضافة، سالب للخصم)'),
                        Textarea::make('description')
                            ->label('الوصف')
                            ->rows(3)
                            ->columnSpanFull()
                            ->placeholder('وصف المعاملة...'),
                        DatePicker::make('expires_at')
                            ->label('تاريخ انتهاء الصلاحية')
                            ->displayFormat('d/m/Y')
                            ->helperText('تاريخ انتهاء صلاحية النقاط (يتم حسابه تلقائياً عند إضافة نقاط جديدة)')
                            ->visible(fn ($get) => in_array($get('type'), ['earned', 'admin_adjustment'])),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('client.name')
                    ->label('العميل')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),
                Tables\Columns\TextColumn::make('type')
                    ->label('النوع')
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'earned' => 'اكتساب',
                        'spent' => 'استخدام',
                        'expired' => 'انتهاء',
                        'admin_adjustment' => 'تعديل إداري',
                        default => $state,
                    })
                    ->color(fn (string $state): string => match ($state) {
                        'earned' => 'success',
                        'spent' => 'danger',
                        'expired' => 'gray',
                        'admin_adjustment' => 'warning',
                        default => 'gray',
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('points')
                    ->label('النقاط')
                    ->formatStateUsing(fn (int $state, PointTransaction $record): string => 
                        ($record->type === 'earned' || $record->type === 'admin_adjustment') ? '+' . $state : '-' . abs($state)
                    )
                    ->badge()
                    ->color(fn (PointTransaction $record): string => 
                        ($record->type === 'earned' || $record->type === 'admin_adjustment') ? 'success' : 'danger'
                    )
                    ->sortable(),
                Tables\Columns\TextColumn::make('balance_before')
                    ->label('الرصيد قبل')
                    ->suffix(' نقطة')
                    ->badge()
                    ->color('gray'),
                Tables\Columns\TextColumn::make('balance_after')
                    ->label('الرصيد بعد')
                    ->suffix(' نقطة')
                    ->badge()
                    ->color('success'),
                Tables\Columns\TextColumn::make('description')
                    ->label('الوصف')
                    ->limit(50)
                    ->searchable(),
                Tables\Columns\TextColumn::make('expires_at')
                    ->label('تاريخ انتهاء الصلاحية')
                    ->date('d/m/Y')
                    ->sortable()
                    ->color(fn (PointTransaction $record): ?string => 
                        $record->expires_at && $record->expires_at->isPast() ? 'danger' : 
                        ($record->expires_at && $record->expires_at->isToday() ? 'warning' : null)
                    )
                    ->badge(fn (PointTransaction $record): bool => 
                        $record->expires_at && ($record->expires_at->isPast() || $record->expires_at->isToday())
                    )
                    ->placeholder('لا يوجد'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('التاريخ')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('type')
                    ->label('نوع المعاملة')
                    ->options([
                        'earned' => 'اكتساب نقاط',
                        'spent' => 'استخدام نقاط',
                        'expired' => 'انتهاء نقاط',
                        'admin_adjustment' => 'تعديل إداري',
                    ]),
                Tables\Filters\SelectFilter::make('client_id')
                    ->label('العميل')
                    ->relationship('client', 'name')
                    ->searchable()
                    ->preload(),
            ])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                DeleteBulkAction::make(),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPointTransactions::route('/'),
            'create' => Pages\CreatePointTransaction::route('/create'),
            'edit' => Pages\EditPointTransaction::route('/{record}/edit'),
        ];
    }
}





