<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PackageResource\Pages;
use App\Models\Package;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;

class PackageResource extends Resource
{
    protected static ?string $model = Package::class;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-gift';

    protected static ?string $navigationLabel = 'باقات الاشتراك';

    protected static ?string $modelLabel = 'باقة';

    protected static ?string $pluralModelLabel = 'باقات الاشتراك';

    protected static string | \UnitEnum | null $navigationGroup = 'إدارة الاشتراكات';

    protected static ?int $navigationSort = 1;

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Tabs::make('Translations')
                    ->tabs([
                        Tab::make('العربية')
                            ->schema([
                                TextInput::make('name.ar')
                                    ->label('اسم الباقة (عربي)')
                                    ->required()
                                    ->maxLength(255)
                                    ->placeholder('مثال: أساسي، فضي، ذهبي'),
                                Textarea::make('description.ar')
                                    ->label('وصف الباقة (عربي)')
                                    ->rows(3)
                                    ->columnSpanFull()
                                    ->placeholder('وصف مختصر عن مميزات الباقة...'),
                                TextInput::make('badge_text.ar')
                                    ->label('نص الشارة (عربي)')
                                    ->maxLength(255)
                                    ->placeholder('مثال: وفّر 20% على كل خدمة'),
                            ]),
                        Tab::make('English')
                            ->schema([
                                TextInput::make('name.en')
                                    ->label('Package Name (English)')
                                    ->required()
                                    ->maxLength(255)
                                    ->placeholder('e.g. Basic, Silver, Gold'),
                                Textarea::make('description.en')
                                    ->label('Description (English)')
                                    ->rows(3)
                                    ->columnSpanFull()
                                    ->placeholder('Brief description of the package...'),
                                TextInput::make('badge_text.en')
                                    ->label('Badge Text (English)')
                                    ->maxLength(255)
                                    ->placeholder('e.g. Save 20% on every service'),
                            ]),
                    ])
                    ->columnSpanFull(),

                Section::make('المظهر')
                    ->components([
                        Select::make('icon')
                            ->label('أيقونة الباقة')
                            ->options([
                                '👑' => '👑 تاج',
                                '⭐' => '⭐ نجمة',
                                '💎' => '💎 ألماس',
                                '🏆' => '🏆 كأس',
                                '🎯' => '🎯 هدف',
                                '🔥' => '🔥 نار',
                                '✨' => '✨ لامع',
                                '🌟' => '🌟 نجمة لامعة',
                                '🛡️' => '🛡️ درع',
                                '🎁' => '🎁 هدية',
                            ])
                            ->native(false)
                            ->searchable(),
                        ColorPicker::make('color')
                            ->label('لون الباقة')
                            ->required()
                            ->helperText('اختر اللون الأساسي للباقة (مثل البرتقالي للأساسي، الذهبي للذهبي)'),
                    ])->columns(2),

                Section::make('التسعير والمدة')
                    ->description('حدد سعر الباقة ومدة الاشتراك')
                    ->components([
                        TextInput::make('price')
                            ->label('السعر')
                            ->numeric()
                            ->required()
                            ->prefix('ر.س')
                            ->step(0.01)
                            ->placeholder('0.00'),
                        Select::make('duration_days')
                            ->label('مدة الاشتراك')
                            ->options([
                                180 => 'نصف سنوي (6 أشهر - 180 يوم)',
                                365 => 'سنوي (12 شهر - 365 يوم)',
                            ])
                            ->required()
                            ->default(180)
                            ->native(false)
                            ->helperText('اختر مدة الاشتراك: نصف سنوي (180 يوم) أو سنوي (365 يوم)'),
                        TextInput::make('discount_percentage')
                            ->label('نسبة الخصم العامة')
                            ->numeric()
                            ->default(0)
                            ->suffix('%')
                            ->minValue(0)
                            ->maxValue(100)
                            ->helperText('نسبة الخصم على جميع الخدمات المشمولة'),
                    ])->columns(2),

                Section::make('الأقسام المشمولة')
                    ->description('اختر الأقسام التي تشملها هذه الباقة مع نسبة الخصم لكل قسم')
                    ->components([
                        Select::make('categories')
                            ->label('الأقسام')
                            ->relationship('categories', 'name')
                            ->getOptionLabelFromRecordUsing(fn ($record) => $record->getTranslation('name', 'ar') . ' / ' . $record->getTranslation('name', 'en'))
                            ->multiple()
                            ->preload()
                            ->searchable()
                            ->columnSpanFull()
                            ->helperText('اختر الأقسام التي يحصل عليها المشترك (مثل: كهرباء، تكييف، سباكة)'),
                    ]),

                Section::make('المميزات الإضافية')
                    ->description('أضف المميزات الإضافية التي تظهر في كارت الباقة')
                    ->components([
                        TextInput::make('free_maintenance_visits')
                            ->label('عدد زيارات الصيانة المجانية')
                            ->numeric()
                            ->default(0)
                            ->suffix('زيارة')
                            ->helperText('عدد الزيارات المجانية للصيانة الدورية'),
                        TextInput::make('warranty_months')
                            ->label('مدة الضمان على العمل')
                            ->numeric()
                            ->default(0)
                            ->suffix('شهر')
                            ->helperText('مدة ضمان الأعمال المنجزة بالأشهر'),
                        Checkbox::make('priority_booking')
                            ->label('أولوية الحجز')
                            ->helperText('المشترك يحصل على أعلى أولوية في حجز المواعيد'),
                        Checkbox::make('support_24_7')
                            ->label('دعم على مدار الساعة 24/7')
                            ->helperText('المشترك يحصل على دعم فني متواصل'),
                    ])->columns(2),

                Section::make('مميزات مخصصة')
                    ->description('أضف أي مميزات إضافية تظهر في بطاقة الباقة')
                    ->components([
                        Repeater::make('features')
                            ->label('المميزات')
                            ->relationship()
                            ->components([
                                TextInput::make('feature.ar')
                                    ->label('الميزة (عربي)')
                                    ->required()
                                    ->maxLength(255)
                                    ->placeholder('مثال: خصم 20% على جميع الخدمات'),
                                TextInput::make('feature.en')
                                    ->label('Feature (English)')
                                    ->required()
                                    ->maxLength(255)
                                    ->placeholder('e.g. 20% discount on all services'),
                                Select::make('icon')
                                    ->label('الأيقونة')
                                    ->options([
                                        '✅' => '✅ صح أخضر',
                                        '⭐' => '⭐ نجمة',
                                        '🔧' => '🔧 مفتاح',
                                        '🛡️' => '🛡️ درع',
                                        '📞' => '📞 هاتف',
                                        '🕐' => '🕐 ساعة',
                                        '💰' => '💰 خصم',
                                        '🎯' => '🎯 هدف',
                                        '🏠' => '🏠 منزل',
                                        '⚡' => '⚡ سريع',
                                    ])
                                    ->default('✅')
                                    ->native(false),
                                TextInput::make('sort_order')
                                    ->label('الترتيب')
                                    ->numeric()
                                    ->default(0),
                            ])
                            ->columns(4)
                            ->defaultItems(0)
                            ->addActionLabel('إضافة ميزة جديدة')
                            ->reorderable()
                            ->collapsible()
                            ->columnSpanFull(),
                    ]),

                Section::make('إعدادات إضافية')
                    ->components([
                        TextInput::make('sort_order')
                            ->label('ترتيب العرض')
                            ->numeric()
                            ->default(0),
                        Checkbox::make('is_active')
                            ->label('الباقة نشطة')
                            ->default(true)
                            ->helperText('هل الباقة متاحة للاشتراك؟'),
                        Checkbox::make('is_featured')
                            ->label('باقة مميزة')
                            ->default(false)
                            ->helperText('الباقة المميزة تظهر بشكل بارز للمستخدمين'),
                    ])->columns(3),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ColorColumn::make('color')
                    ->label('اللون'),
                Tables\Columns\TextColumn::make('name')
                    ->label('اسم الباقة')
                    ->formatStateUsing(fn ($record) => $record->getTranslation('name', 'ar') . ' / ' . $record->getTranslation('name', 'en'))
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->icon(fn (Package $record): ?string => $record->is_featured ? 'heroicon-s-star' : null)
                    ->iconColor('warning'),
                Tables\Columns\TextColumn::make('price')
                    ->label('السعر')
                    ->money('SAR')
                    ->sortable(),
                Tables\Columns\TextColumn::make('discount_percentage')
                    ->label('نسبة الخصم')
                    ->suffix('%')
                    ->badge()
                    ->color('success')
                    ->sortable(),
                Tables\Columns\TextColumn::make('duration_days')
                    ->label('المدة')
                    ->formatStateUsing(fn (int $state): string => match ($state) {
                        180 => 'نصف سنوي (180 يوم)',
                        365 => 'سنوي (365 يوم)',
                        default => $state . ' يوم',
                    })
                    ->badge()
                    ->color(fn (int $state): string => match ($state) {
                        180 => 'info',
                        365 => 'success',
                        default => 'gray',
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('categories.name')
                    ->label('الأقسام المشمولة')
                    ->formatStateUsing(fn ($record) => $record->categories->map(fn ($cat) => $cat->getTranslation('name', 'ar'))->join(', '))
                    ->badge()
                    ->color('info'),
                Tables\Columns\TextColumn::make('features_count')
                    ->label('المميزات')
                    ->counts('features')
                    ->badge()
                    ->color('warning'),
                Tables\Columns\TextColumn::make('subscriptions_count')
                    ->label('عدد المشتركين')
                    ->counts('subscriptions')
                    ->badge()
                    ->color('info')
                    ->sortable(),
                Tables\Columns\IconColumn::make('priority_booking')
                    ->label('أولوية')
                    ->boolean(),
                Tables\Columns\IconColumn::make('support_24_7')
                    ->label('دعم 24/7')
                    ->boolean(),
                Tables\Columns\IconColumn::make('is_active')
                    ->label('نشط')
                    ->boolean(),
                Tables\Columns\IconColumn::make('is_featured')
                    ->label('مميز')
                    ->boolean()
                    ->trueIcon('heroicon-s-star')
                    ->falseIcon('heroicon-o-star')
                    ->trueColor('warning'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->placeholder('الكل')
                    ->trueLabel('نشط فقط')
                    ->falseLabel('غير نشط فقط'),
                Tables\Filters\TernaryFilter::make('is_featured')
                    ->label('مميز')
                    ->placeholder('الكل')
                    ->trueLabel('مميز فقط')
                    ->falseLabel('غير مميز فقط'),
            ])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                DeleteBulkAction::make(),
            ])
            ->defaultSort('sort_order');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPackages::route('/'),
            'create' => Pages\CreatePackage::route('/create'),
            'edit' => Pages\EditPackage::route('/{record}/edit'),
        ];
    }
}
