<?php

namespace App\Filament\Resources\OrderResource\Pages;

use App\Filament\Resources\OrderResource;
use Filament\Actions;
use Filament\Actions\Action;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ViewRecord;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Illuminate\Support\Carbon;

class ViewOrder extends ViewRecord
{

    protected static string $resource = OrderResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\EditAction::make(),
            Actions\DeleteAction::make(),
            Action::make('change_status')
                ->label('تغيير الحالة')
                ->icon('heroicon-o-arrow-path')
                ->color('warning')
                ->form([
                    Select::make('status')
                        ->label('الحالة الجديدة')
                        ->options([
                            'pending' => 'قيد الانتظار',
                            'confirmed' => 'مؤكد',
                            'in_progress' => 'قيد التنفيذ',
                            'completed' => 'مكتمل',
                            'cancelled' => 'ملغي',
                        ])
                        ->required()
                        ->default($this->record->status)
                        ->native(false),
                ])
                ->action(function (array $data) {
                    $this->record->update([
                        'status' => $data['status'],
                        'completed_date' => $data['status'] === 'completed' && !$this->record->completed_date 
                            ? Carbon::now() 
                            : $this->record->completed_date,
                    ]);

                    Notification::make()
                        ->title('تم تغيير الحالة بنجاح')
                        ->success()
                        ->send();
                }),
            Action::make('add_note')
                ->label('إضافة ملاحظة')
                ->icon('heroicon-o-plus-circle')
                ->color('info')
                ->form([
                    Textarea::make('note')
                        ->label('الملاحظة')
                        ->required()
                        ->rows(4)
                        ->placeholder('أدخل الملاحظة هنا...'),
                ])
                ->action(function (array $data) {
                    $currentNotes = $this->record->notes ?? '';
                    $newNote = "\n[" . Carbon::now()->format('Y-m-d H:i') . "] " . $data['note'];
                    
                    $this->record->update([
                        'notes' => $currentNotes . $newNote,
                    ]);

                    Notification::make()
                        ->title('تم إضافة الملاحظة بنجاح')
                        ->success()
                        ->send();
                }),
            Action::make('edit_pricing')
                ->label('تعديل الأسعار والمستحق')
                ->icon('heroicon-o-currency-dollar')
                ->color('success')
                ->form([
                    TextInput::make('price')
                        ->label('السعر')
                        ->numeric()
                        ->prefix('ر.س')
                        ->required()
                        ->default(fn () => $this->record->price)
                        ->live()
                        ->afterStateUpdated(function ($set, $get) {
                            $price = $get('price') ?? 0;
                            $fee = $get('fee') ?? 0;
                            $discount = $get('discount') ?? 0;
                            $set('total', $price + $fee - $discount);
                        }),
                    TextInput::make('fee')
                        ->label('رسوم التطبيق')
                        ->numeric()
                        ->prefix('ر.س')
                        ->required()
                        ->default(fn () => $this->record->fee)
                        ->live()
                        ->afterStateUpdated(function ($set, $get) {
                            $price = $get('price') ?? 0;
                            $fee = $get('fee') ?? 0;
                            $discount = $get('discount') ?? 0;
                            $set('total', $price + $fee - $discount);
                        }),
                    TextInput::make('technician_commission')
                        ->label('مستحق الفني')
                        ->numeric()
                        ->prefix('ر.س')
                        ->default(fn () => $this->record->technician_commission ?? 0)
                        ->helperText('المبلغ المستحق للفني من هذا الطلب'),
                    TextInput::make('discount')
                        ->label('الخصم')
                        ->numeric()
                        ->prefix('ر.س')
                        ->default(fn () => $this->record->discount ?? 0)
                        ->live()
                        ->afterStateUpdated(function ($set, $get) {
                            $price = $get('price') ?? 0;
                            $fee = $get('fee') ?? 0;
                            $discount = $get('discount') ?? 0;
                            $set('total', $price + $fee - $discount);
                        }),
                    TextInput::make('total')
                        ->label('الإجمالي')
                        ->numeric()
                        ->prefix('ر.س')
                        ->disabled()
                        ->dehydrated()
                        ->default(fn () => $this->record->total),
                ])
                ->action(function (array $data) {
                    // حساب الإجمالي تلقائياً
                    $total = ($data['price'] ?? 0) + ($data['fee'] ?? 0) - ($data['discount'] ?? 0);
                    
                    $this->record->update([
                        'price' => $data['price'],
                        'fee' => $data['fee'],
                        'technician_commission' => $data['technician_commission'] ?? 0,
                        'discount' => $data['discount'] ?? 0,
                        'total' => $total,
                    ]);

                    Notification::make()
                        ->title('تم تحديث الأسعار والمستحق بنجاح')
                        ->success()
                        ->send();
                }),
        ];
    }

}

