<?php

namespace App\Filament\Resources;

use App\Filament\Resources\OrderResource\Pages;
use App\Models\Client;
use App\Models\Order;
use App\Models\Service;
use App\Models\Technician;
use App\Settings\GeneralSettings;
use Spatie\LaravelSettings\Exceptions\MissingSettings;
use Filament\Actions\BulkAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;

class OrderResource extends Resource
{
    protected static ?string $model = Order::class;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-shopping-cart';

    protected static ?string $navigationLabel = 'الطلبات';

    protected static ?string $modelLabel = 'طلب';

    protected static ?string $pluralModelLabel = 'الطلبات';

    protected static string | \UnitEnum | null $navigationGroup = 'إدارة الطلبات';

    protected static ?int $navigationSort = 1;

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('معلومات الطلب')
                    ->components([
                        TextInput::make('order_number')
                            ->label('رقم الطلب')
                            ->disabled()
                            ->dehydrated()
                            ->default('ORD-' . strtoupper(\Illuminate\Support\Str::random(8)))
                            ->columnSpanFull(),
                        Select::make('client_id')
                            ->label('العميل')
                            ->relationship('client', 'name')
                            ->required()
                            ->searchable()
                            ->preload()
                            ->columnSpan(1),
                        Select::make('service_id')
                            ->label('الخدمة')
                            ->relationship('service', 'name')
                            ->getOptionLabelFromRecordUsing(fn ($record) => $record->getTranslation('name', 'ar'))
                            ->required()
                            ->searchable()
                            ->preload()
                            ->columnSpan(1),
                        Select::make('technician_id')
                            ->label('الفني')
                            ->relationship('technician', 'name')
                            ->required()
                            ->searchable()
                            ->preload()
                            ->columnSpan(1),
                        Select::make('type')
                            ->label('نوع الطلب')
                            ->options([
                                'normal' => 'عادي',
                                'urgent' => 'مستعجل (24 ساعة)',
                            ])
                            ->required()
                            ->default('normal')
                            ->native(false)
                            ->columnSpan(1),
                        Select::make('status')
                            ->label('الحالة')
                            ->options([
                                'pending' => 'قيد الانتظار',
                                'confirmed' => 'مؤكد',
                                'in_progress' => 'قيد التنفيذ',
                                'completed' => 'مكتمل',
                                'cancelled' => 'ملغي',
                            ])
                            ->required()
                            ->default('pending')
                            ->native(false)
                            ->columnSpan(1),
                    ])->columns(2),
                Section::make('التواريخ')
                    ->components([
                        DateTimePicker::make('scheduled_date')
                            ->label('تاريخ الحجز')
                            ->native(false)
                            ->displayFormat('Y-m-d H:i')
                            ->columnSpan(1),
                        DateTimePicker::make('completed_date')
                            ->label('تاريخ الإتمام')
                            ->native(false)
                            ->displayFormat('Y-m-d H:i')
                            ->columnSpan(1),
                    ])->columns(2),
                Section::make('التسعير')
                    ->components([
                        TextInput::make('price')
                            ->label('السعر')
                            ->numeric()
                            ->prefix('ر.س')
                            ->required()
                            ->live()
                            ->afterStateUpdated(fn ($set, $get) => $set('total', ($get('price') ?? 0) + ($get('fee') ?? 0) - ($get('discount') ?? 0)))
                            ->columnSpan(1),
                        TextInput::make('fee')
                            ->label('رسوم التطبيق')
                            ->numeric()
                            ->prefix('ر.س')
                            ->default(function () {
                                try {
                                    return app(GeneralSettings::class)->apps_fee ?? 0;
                                } catch (MissingSettings $e) {
                                    return 0;
                                }
                            })
                            ->live()
                            ->afterStateUpdated(fn ($set, $get) => $set('total', ($get('price') ?? 0) + ($get('fee') ?? 0) - ($get('discount') ?? 0)))
                            ->helperText('رسوم التطبيق - القيمة الافتراضية من الإعدادات')
                            ->columnSpan(1),
                        TextInput::make('technician_commission')
                            ->label('مستحق الفني')
                            ->numeric()
                            ->prefix('ر.س')
                            ->default(0)
                            ->helperText('المبلغ المستحق للفني من هذا الطلب')
                            ->columnSpan(1),
                        TextInput::make('discount')
                            ->label('الخصم')
                            ->numeric()
                            ->prefix('ر.س')
                            ->default(0)
                            ->live()
                            ->afterStateUpdated(fn ($set, $get) => $set('total', ($get('price') ?? 0) + ($get('fee') ?? 0) - ($get('discount') ?? 0)))
                            ->columnSpan(1),
                        TextInput::make('total')
                            ->label('الإجمالي')
                            ->numeric()
                            ->prefix('ر.س')
                            ->required()
                            ->disabled()
                            ->dehydrated()
                            ->default(fn ($get) => ($get('price') ?? 0) + ($get('fee') ?? 0) - ($get('discount') ?? 0))
                            ->columnSpan(1),
                    ])->columns(2),
                Section::make('العنوان')
                    ->components([
                        Textarea::make('address')
                            ->label('العنوان')
                            ->rows(2)
                            ->columnSpanFull(),
                        TextInput::make('city')
                            ->label('المدينة')
                            ->maxLength(255)
                            ->columnSpanFull(),
                    ]),
                Section::make('ملاحظات')
                    ->components([
                        Textarea::make('notes')
                            ->label('ملاحظات إدارية')
                            ->rows(3)
                            ->columnSpanFull(),
                        Textarea::make('client_notes')
                            ->label('ملاحظات العميل')
                            ->rows(3)
                            ->columnSpanFull(),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(fn ($query) => $query->with(['client', 'service', 'technician', 'rating']))
            ->columns([
                Tables\Columns\TextColumn::make('order_number')
                    ->label('رقم الطلب')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),
                Tables\Columns\TextColumn::make('client.name')
                    ->label('العميل')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('service.name')
                    ->label('الخدمة')
                    ->formatStateUsing(fn ($record) => $record->service ? $record->service->getTranslation('name', 'ar') : '-')
                    ->searchable(),
                Tables\Columns\TextColumn::make('technician.name')
                    ->label('الفني')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('type')
                    ->label('النوع')
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'urgent' => 'مستعجل',
                        'normal' => 'عادي',
                        default => $state,
                    })
                    ->color(fn (string $state): string => match ($state) {
                        'urgent' => 'danger',
                        'normal' => 'info',
                        default => 'gray',
                    }),
                Tables\Columns\TextColumn::make('status')
                    ->label('الحالة')
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'pending' => 'قيد الانتظار',
                        'confirmed' => 'مؤكد',
                        'in_progress' => 'قيد التنفيذ',
                        'completed' => 'مكتمل',
                        'cancelled' => 'ملغي',
                        default => $state,
                    })
                    ->color(fn (string $state): string => match ($state) {
                        'pending' => 'warning',
                        'confirmed' => 'info',
                        'in_progress' => 'primary',
                        'completed' => 'success',
                        'cancelled' => 'danger',
                        default => 'gray',
                    }),
                Tables\Columns\TextColumn::make('scheduled_date')
                    ->label('تاريخ الحجز')
                    ->dateTime('Y-m-d H:i')
                    ->sortable(),
                Tables\Columns\TextColumn::make('total')
                    ->label('الإجمالي')
                    ->money('SAR')
                    ->sortable(),
                Tables\Columns\TextColumn::make('technician_commission')
                    ->label('مستحق الفني')
                    ->money('SAR')
                    ->sortable()
                    ->toggleable(),
                Tables\Columns\IconColumn::make('rating')
                    ->label('التقييم')
                    ->icon(fn ($record) => $record->rating ? 'heroicon-o-star' : null)
                    ->color(fn ($record) => $record->rating ? 'warning' : null)
                    ->tooltip(fn ($record) => $record->rating ? 'التقييم: ' . $record->rating->rating . '/5' : 'لا يوجد تقييم'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('type')
                    ->label('نوع الطلب')
                    ->options([
                        'normal' => 'عادي',
                        'urgent' => 'مستعجل',
                    ])
                    ->placeholder('كل الأنواع'),
                Tables\Filters\SelectFilter::make('status')
                    ->label('الحالة')
                    ->options([
                        'pending' => 'قيد الانتظار',
                        'confirmed' => 'مؤكد',
                        'in_progress' => 'قيد التنفيذ',
                        'completed' => 'مكتمل',
                        'cancelled' => 'ملغي',
                    ])
                    ->placeholder('كل الحالات'),
                Tables\Filters\Filter::make('urgent_orders')
                    ->label('🚨 طلبات مستعجلة')
                    ->query(fn ($query) => $query->where('type', 'urgent')),
                Tables\Filters\Filter::make('completed_without_rating')
                    ->label('✅ مكتملة بدون تقييم')
                    ->query(fn ($query) => $query->where('status', 'completed')
                        ->whereDoesntHave('rating')),
            ])
            ->filtersFormColumns(2)
            ->actions([
                ViewAction::make(),
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                DeleteBulkAction::make(),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function infolist(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('معلومات الطلب')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('order_number')
                            ->label('رقم الطلب')
                            ->badge()
                            ->color('primary')
                            ->size('lg'),
                        \Filament\Infolists\Components\TextEntry::make('client.name')
                            ->label('العميل')
                            ->icon('heroicon-o-user'),
                        \Filament\Infolists\Components\TextEntry::make('service.name')
                            ->label('الخدمة')
                            ->formatStateUsing(fn ($state, $record) => $record->service ? $record->service->getTranslation('name', 'ar') : '-')
                            ->icon('heroicon-o-wrench-screwdriver'),
                        \Filament\Infolists\Components\TextEntry::make('technician.name')
                            ->label('الفني')
                            ->icon('heroicon-o-user-circle'),
                        \Filament\Infolists\Components\TextEntry::make('type')
                            ->label('نوع الطلب')
                            ->badge()
                            ->formatStateUsing(fn (string $state): string => match ($state) {
                                'urgent' => 'مستعجل (24 ساعة)',
                                'normal' => 'عادي',
                                default => $state,
                            })
                            ->color(fn (string $state): string => match ($state) {
                                'urgent' => 'danger',
                                'normal' => 'info',
                                default => 'gray',
                            }),
                        \Filament\Infolists\Components\TextEntry::make('status')
                            ->label('الحالة')
                            ->badge()
                            ->formatStateUsing(fn (string $state): string => match ($state) {
                                'pending' => 'قيد الانتظار',
                                'confirmed' => 'مؤكد',
                                'in_progress' => 'قيد التنفيذ',
                                'completed' => 'مكتمل',
                                'cancelled' => 'ملغي',
                                default => $state,
                            })
                            ->color(fn (string $state): string => match ($state) {
                                'pending' => 'warning',
                                'confirmed' => 'info',
                                'in_progress' => 'primary',
                                'completed' => 'success',
                                'cancelled' => 'danger',
                                default => 'gray',
                            }),
                    ])->columns(2),
                Section::make('التواريخ')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('scheduled_date')
                            ->label('تاريخ الحجز')
                            ->formatStateUsing(fn ($state) => $state ? $state->format('Y-m-d H:i') : '-')
                            ->icon('heroicon-o-calendar'),
                        \Filament\Infolists\Components\TextEntry::make('completed_date')
                            ->label('تاريخ الإتمام')
                            ->formatStateUsing(fn ($state) => $state ? $state->format('Y-m-d H:i') : '-')
                            ->icon('heroicon-o-check-circle'),
                        \Filament\Infolists\Components\TextEntry::make('created_at')
                            ->label('تاريخ الإنشاء')
                            ->dateTime('Y-m-d H:i')
                            ->icon('heroicon-o-clock'),
                    ])->columns(3),
                Section::make('التسعير')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('price')
                            ->label('السعر')
                            ->money('SAR')
                            ->icon('heroicon-o-banknotes'),
                        \Filament\Infolists\Components\TextEntry::make('fee')
                            ->label('رسوم التطبيق')
                            ->money('SAR')
                            ->default('0.00')
                            ->icon('heroicon-o-credit-card'),
                        \Filament\Infolists\Components\TextEntry::make('technician_commission')
                            ->label('مستحق الفني')
                            ->money('SAR')
                            ->default('0.00')
                            ->icon('heroicon-o-user-circle'),
                        \Filament\Infolists\Components\TextEntry::make('discount')
                            ->label('الخصم')
                            ->money('SAR')
                            ->default('0.00')
                            ->icon('heroicon-o-tag'),
                        \Filament\Infolists\Components\TextEntry::make('total')
                            ->label('الإجمالي')
                            ->money('SAR')
                            ->size('lg')
                            ->weight('bold')
                            ->color('success')
                            ->icon('heroicon-o-currency-dollar'),
                    ])->columns(2),
                Section::make('العنوان')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('address')
                            ->label('العنوان')
                            ->default('-')
                            ->icon('heroicon-o-map-pin'),
                        \Filament\Infolists\Components\TextEntry::make('city')
                            ->label('المدينة')
                            ->default('-')
                            ->icon('heroicon-o-building-office'),
                    ])->columns(2),
                Section::make('ملاحظات')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('notes')
                            ->label('ملاحظات إدارية')
                            ->default('لا توجد ملاحظات')
                            ->formatStateUsing(function ($state) {
                                if (empty($state) || $state === 'لا توجد ملاحظات') {
                                    return 'لا توجد ملاحظات';
                                }
                                
                                // تنظيف النص من المسافات الزائدة في البداية
                                $state = trim($state);
                                
                                // تقسيم الملاحظات بناءً على pattern [YYYY-MM-DD HH:MM] أو [HH:MM YYYY-MM-DD]
                                // نبحث عن أي timestamp pattern
                                $pattern = '/(\[\d{4}-\d{2}-\d{2} \d{2}:\d{2}\]|\[\d{2}:\d{2} \d{4}-\d{2}-\d{2}\])/';
                                
                                // تقسيم النص بناءً على timestamps
                                $parts = preg_split($pattern, $state, -1, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);
                                
                                $formatted = '';
                                $currentNote = '';
                                
                                foreach ($parts as $part) {
                                    $part = trim($part);
                                    if (empty($part)) continue;
                                    
                                    // التحقق إذا كان هذا timestamp
                                    if (preg_match($pattern, $part)) {
                                        // إذا كان هناك ملاحظة سابقة، أضفها
                                        if (!empty($currentNote)) {
                                            $formatted .= trim($currentNote) . "\n\n";
                                            $currentNote = '';
                                        }
                                        // أضف timestamp كعنوان
                                        $formatted .= "**" . $part . "**\n";
                                    } else {
                                        // هذا جزء من الملاحظة
                                        $currentNote .= ($currentNote ? ' ' : '') . $part;
                                    }
                                }
                                
                                // أضف آخر ملاحظة إذا كانت موجودة
                                if (!empty($currentNote)) {
                                    $formatted .= trim($currentNote);
                                }
                                
                                return trim($formatted) ?: 'لا توجد ملاحظات';
                            })
                            ->markdown()
                            ->columnSpanFull(),
                        \Filament\Infolists\Components\TextEntry::make('client_notes')
                            ->label('ملاحظات العميل')
                            ->default('لا توجد ملاحظات')
                            ->markdown()
                            ->columnSpanFull(),
                    ]),
                Section::make('التقييم')
                    ->schema([
                        \Filament\Infolists\Components\TextEntry::make('rating.rating')
                            ->label('التقييم')
                            ->formatStateUsing(fn ($state) => $state ? str_repeat('⭐', $state) . str_repeat('☆', 5 - $state) . ' (' . $state . '/5)' : 'لا يوجد تقييم')
                            ->badge()
                            ->color(fn ($state) => $state ? match ($state) {
                                5, 4 => 'success',
                                3 => 'warning',
                                1, 2 => 'danger',
                                default => 'gray',
                            } : 'gray'),
                        \Filament\Infolists\Components\TextEntry::make('rating.comment')
                            ->label('تعليق العميل')
                            ->default('-')
                            ->columnSpanFull(),
                        \Filament\Infolists\Components\TextEntry::make('rating.created_at')
                            ->label('تاريخ التقييم')
                            ->formatStateUsing(fn ($state) => $state ? $state->format('Y-m-d H:i') : '-'),
                    ])
                    ->visible(fn ($record) => $record->rating)
                    ->columns(2),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListOrders::route('/'),
            'create' => Pages\CreateOrder::route('/create'),
            'view' => Pages\ViewOrder::route('/{record}'),
            'edit' => Pages\EditOrder::route('/{record}/edit'),
        ];
    }
}

