<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CategoryResource\Pages;
use App\Models\Category;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Str;

class CategoryResource extends Resource
{
    protected static ?string $model = Category::class;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-squares-2x2';

    protected static ?string $navigationLabel = 'الأقسام';

    protected static ?string $modelLabel = 'قسم';

    protected static ?string $pluralModelLabel = 'الأقسام';

    protected static string | \UnitEnum | null $navigationGroup = 'إدارة الخدمات';

    protected static ?int $navigationSort = 0;

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Tabs::make('Translations')
                    ->tabs([
                        Tab::make('العربية')
                            ->schema([
                                TextInput::make('name.ar')
                                    ->label('اسم القسم (عربي)')
                                    ->required()
                                    ->maxLength(255),
                                Textarea::make('description.ar')
                                    ->label('الوصف (عربي)')
                                    ->rows(3)
                                    ->columnSpanFull(),
                            ]),
                        Tab::make('English')
                            ->schema([
                                TextInput::make('name.en')
                                    ->label('Category Name (English)')
                                    ->required()
                                    ->maxLength(255),
                                Textarea::make('description.en')
                                    ->label('Description (English)')
                                    ->rows(3)
                                    ->columnSpanFull(),
                            ]),
                    ])
                    ->columnSpanFull(),
                Section::make('معلومات إضافية')
                    ->components([
                        Select::make('icon')
                            ->label('الأيقونة')
                            ->options([
                                'heroicon-o-home' => '🏠 المنزل',
                                'heroicon-o-wrench-screwdriver' => '🔧 صيانة عامة',
                                'heroicon-o-bolt' => '⚡ كهرباء',
                                'heroicon-o-fire' => '🔥 تدفئة/غاز',
                                'heroicon-o-beaker' => '💧 سباكة',
                                'heroicon-o-paint-brush' => '🎨 دهان',
                                'heroicon-o-cube' => '📦 نجارة',
                                'heroicon-o-cog-6-tooth' => '⚙️ تكييف',
                                'heroicon-o-tv' => '📺 أجهزة كهربائية',
                                'heroicon-o-computer-desktop' => '💻 تقنية',
                                'heroicon-o-sparkles' => '✨ تنظيف',
                                'heroicon-o-shield-check' => '🛡️ أمان',
                                'heroicon-o-key' => '🔑 أقفال',
                                'heroicon-o-truck' => '🚚 نقل',
                                'heroicon-o-scissors' => '✂️ حدائق',
                                'heroicon-o-sun' => '☀️ طاقة شمسية',
                            ])
                            ->searchable()
                            ->native(false)
                            ->helperText('اختر أيقونة تمثل هذا القسم'),
                        Checkbox::make('is_active')
                            ->label('نشط')
                            ->default(true),
                        TextInput::make('sort_order')
                            ->label('ترتيب العرض')
                            ->numeric()
                            ->default(0),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\IconColumn::make('icon')
                    ->label('الأيقونة')
                    ->icon(fn (string $state): string => $state ?: 'heroicon-o-squares-2x2')
                    ->color('primary'),
                Tables\Columns\TextColumn::make('name')
                    ->label('اسم القسم')
                    ->formatStateUsing(fn ($record) => $record->getTranslation('name', 'ar') . ' / ' . $record->getTranslation('name', 'en'))
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('description')
                    ->label('الوصف')
                    ->formatStateUsing(fn ($record) => $record->getTranslation('description', 'ar') ?: '')
                    ->limit(50)
                    ->searchable(),
                Tables\Columns\IconColumn::make('is_active')
                    ->label('نشط')
                    ->boolean(),
                Tables\Columns\TextColumn::make('sort_order')
                    ->label('الترتيب')
                    ->sortable(),
                Tables\Columns\TextColumn::make('services_count')
                    ->label('عدد الخدمات')
                    ->counts('services')
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('نشط')
                    ->placeholder('الكل')
                    ->trueLabel('نشط فقط')
                    ->falseLabel('غير نشط فقط'),
            ])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                DeleteBulkAction::make(),
            ])
            ->defaultSort('sort_order');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCategories::route('/'),
            'create' => Pages\CreateCategory::route('/create'),
            'edit' => Pages\EditCategory::route('/{record}/edit'),
        ];
    }
}

