<?php

namespace App\Filament\Resources;

use App\Filament\Resources\AdditionalFacilityResource\Pages;
use App\Models\AdditionalFacility;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;

class AdditionalFacilityResource extends Resource
{
    protected static ?string $model = AdditionalFacility::class;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-building-office-2';

    protected static ?string $navigationLabel = 'المرافق الإضافية';

    protected static ?string $modelLabel = 'مرفق إضافي';

    protected static ?string $pluralModelLabel = 'المرافق الإضافية';

    protected static string | \UnitEnum | null $navigationGroup = 'إدارة الاشتراكات';

    protected static ?int $navigationSort = 5;

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Tabs::make('Translations')
                    ->tabs([
                        Tab::make('العربية')
                            ->schema([
                                TextInput::make('name.ar')
                                    ->label('اسم المرفق (عربي)')
                                    ->required()
                                    ->maxLength(255),
                            ]),
                        Tab::make('English')
                            ->schema([
                                TextInput::make('name.en')
                                    ->label('Facility Name (English)')
                                    ->required()
                                    ->maxLength(255),
                            ]),
                    ])
                    ->columnSpanFull(),

                Section::make('التسعير والإعدادات')
                    ->components([
                        TextInput::make('price')
                            ->label('السعر')
                            ->numeric()
                            ->prefix('ر.س')
                            ->step(0.01)
                            ->default(0)
                            ->required()
                            ->helperText('سعر المرفق الإضافي بالريال السعودي'),
                        TextInput::make('icon')
                            ->label('الأيقونة')
                            ->maxLength(255)
                            ->placeholder('🛏️')
                            ->helperText('أيقونة تمثل المرفق (يمكن استخدام emoji)'),
                        TextInput::make('sort_order')
                            ->label('ترتيب العرض')
                            ->numeric()
                            ->default(0)
                            ->helperText('رقم أقل = يظهر أولاً في القائمة'),
                        Checkbox::make('is_active')
                            ->label('نشط')
                            ->default(true)
                            ->helperText('تفعيل/تعطيل ظهور المرفق في النظام'),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('اسم المرفق')
                    ->formatStateUsing(fn ($record) => $record->getTranslation('name', 'ar') . ' / ' . $record->getTranslation('name', 'en'))
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('price')
                    ->label('السعر')
                    ->money('SAR')
                    ->sortable(),
                Tables\Columns\TextColumn::make('icon')
                    ->label('الأيقونة'),
                Tables\Columns\IconColumn::make('is_active')
                    ->label('نشط')
                    ->boolean(),
                Tables\Columns\TextColumn::make('sort_order')
                    ->label('الترتيب')
                    ->sortable(),
                Tables\Columns\TextColumn::make('subscriptions_count')
                    ->label('عدد الاشتراكات')
                    ->counts('subscriptions')
                    ->badge()
                    ->color('info'),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->placeholder('الكل')
                    ->trueLabel('نشط فقط')
                    ->falseLabel('غير نشط فقط'),
            ])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                DeleteBulkAction::make(),
            ])
            ->defaultSort('sort_order');
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListAdditionalFacilities::route('/'),
            'create' => Pages\CreateAdditionalFacility::route('/create'),
            'edit' => Pages\EditAdditionalFacility::route('/{record}/edit'),
        ];
    }
}



