<?php

namespace App\Filament\Pages;

use App\Settings\GeneralSettings;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Actions;
use Filament\Schemas\Components\EmbeddedSchema;
use Filament\Schemas\Components\Form;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Concerns\InteractsWithSchemas;
use Filament\Schemas\Contracts\HasSchemas;
use Filament\Schemas\Schema;
use Spatie\LaravelSettings\Exceptions\MissingSettings;

class Settings extends Page implements HasSchemas
{
    use InteractsWithSchemas;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-cog-6-tooth';

    protected static ?string $navigationLabel = 'الإعدادات';

    protected static string | \UnitEnum | null $navigationGroup = 'إدارة النظام';

    protected static ?int $navigationSort = 10;

    // Form data properties
    public ?float $points_to_riyal = null;
    public ?float $apps_fee = null;
    public ?string $site_name = null;
    public ?string $site_description = null;
    public ?string $site_email = null;
    public ?string $site_phone = null;
    public ?string $site_address = null;
    public ?int $points_expiry_days = null;
    public ?string $order_number_format = null;
    public ?string $order_format_preset = null;

    public function mount(): void
    {
        try {
            $settings = app(GeneralSettings::class);
            // إعدادات التحويل
            $this->points_to_riyal = $settings->points_to_riyal ?? 1;
            $this->apps_fee = $settings->apps_fee ?? 0;
            
            // إعدادات التطبيق
            $this->site_name = $settings->site_name ?? 'بيتي';
            $this->site_description = $settings->site_description ?? 'منصة خدمات منزلية متكاملة';
            $this->site_email = $settings->site_email ?? 'info@bayti.com';
            $this->site_phone = $settings->site_phone ?? '+966500000000';
            $this->site_address = $settings->site_address ?? '';
            $this->points_expiry_days = $settings->points_expiry_days ?? 365;
            $this->order_number_format = $settings->order_number_format ?? 'ORD-{RANDOM:8}';
        } catch (MissingSettings $e) {
            // إذا كانت بعض الإعدادات مفقودة، استخدم القيم الافتراضية
            $this->points_to_riyal = 1;
            $this->apps_fee = 0;
            $this->site_name = 'بيتي';
            $this->site_description = 'منصة خدمات منزلية متكاملة';
            $this->site_email = 'info@bayti.com';
            $this->site_phone = '+966500000000';
            $this->site_address = '';
            $this->points_expiry_days = 365;
            $this->order_number_format = 'ORD-{RANDOM:8}';
        }
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('إعدادات التحويل')
                    ->description('إعدادات تحويل النقاط')
                    ->icon('heroicon-o-currency-dollar')
                    ->components([
                        TextInput::make('points_to_riyal')
                            ->label('عدد النقاط لكل ريال سعودي')
                            ->numeric()
                            ->required()
                            ->default(1)
                            ->helperText('مثال: إذا كانت القيمة 1، فكل نقطة = 1 ريال سعودي')
                            ->columnSpanFull(),
                        TextInput::make('apps_fee')
                            ->label('رسوم التطبيق (ر.س)')
                            ->numeric()
                            ->required()
                            ->default(0)
                            ->prefix('ر.س')
                            ->helperText('رسوم التطبيق الافتراضية التي سيتم تطبيقها على كل طلب')
                            ->columnSpanFull(),
                        TextInput::make('points_expiry_days')
                            ->label('مدة صلاحية النقاط (بالأيام)')
                            ->numeric()
                            ->required()
                            ->default(365)
                            ->suffix('يوم')
                            ->minValue(1)
                            ->helperText('عدد الأيام قبل انتهاء صلاحية النقاط المكتسبة (مثال: 365 يوم = سنة واحدة)')
                            ->columnSpanFull(),
                    ]),

                Section::make('إعدادات التطبيق')
                    ->description('معلومات التطبيق العامة')
                    ->icon('heroicon-o-device-phone-mobile')
                    ->components([
                        TextInput::make('site_name')
                            ->label('اسم التطبيق')
                            ->required()
                            ->maxLength(255),
                        Textarea::make('site_description')
                            ->label('وصف التطبيق')
                            ->rows(3)
                            ->columnSpanFull(),
                        TextInput::make('site_email')
                            ->label('البريد الإلكتروني للتطبيق')
                            ->email()
                            ->maxLength(255),
                        TextInput::make('site_phone')
                            ->label('رقم الهاتف')
                            ->tel()
                            ->maxLength(255),
                        Textarea::make('site_address')
                            ->label('عنوان التطبيق')
                            ->rows(2)
                            ->columnSpanFull(),
                    ])->columns(2),

                Section::make('إعدادات الطلبات')
                    ->description('صيغة رقم الطلب - اختر صيغة جاهزة أو أدخل صيغة مخصصة')
                    ->icon('heroicon-o-shopping-cart')
                    ->components([
                        Select::make('order_format_preset')
                            ->label('صيغ جاهزة (اختياري)')
                            ->options([
                                'ORD-{RANDOM:8}' => 'ORD-{RANDOM:8} → مثال: ORD-12345678',
                                'ORD-{YEAR}{MONTH}{DAY}-{RANDOM:6}' => 'ORD-{YEAR}{MONTH}{DAY}-{RANDOM:6} → مثال: ORD-20240220-123456',
                                'ORD-{YEAR}-{SEQUENCE}' => 'ORD-{YEAR}-{SEQUENCE} → مثال: ORD-2024-000001',
                                'ORD-{YEAR}{MONTH}-{RANDOM:8}' => 'ORD-{YEAR}{MONTH}-{RANDOM:8} → مثال: ORD-202402-12345678',
                                'ORD-{RANDOM:10}' => 'ORD-{RANDOM:10} → مثال: ORD-1234567890',
                                'ORDER-{YEAR}{MONTH}{DAY}{RANDOM:4}' => 'ORDER-{YEAR}{MONTH}{DAY}{RANDOM:4} → مثال: ORDER-202402201234',
                            ])
                            ->placeholder('اختر صيغة جاهزة من القائمة...')
                            ->searchable()
                            ->native(false)
                            ->preload()
                            ->reactive()
                            ->afterStateUpdated(function ($state, callable $set) {
                                if ($state) {
                                    $set('order_number_format', $state);
                                }
                            })
                            ->helperText('يمكنك البحث واختيار صيغة جاهزة من القائمة المنسدلة')
                            ->columnSpanFull(),
                        
                        TextInput::make('order_number_format')
                            ->label('صيغة رقم الطلب')
                            ->required()
                            ->default('ORD-{RANDOM:8}')
                            ->placeholder('أدخل الصيغة المخصصة...')
                            ->helperText('
المتغيرات المتاحة:
• {YEAR} - السنة (4 أرقام) مثال: 2024
• {MONTH} - الشهر (2 رقم) مثال: 02
• {DAY} - اليوم (2 رقم) مثال: 20
• {RANDOM:N} - أرقام عشوائية (N = عدد الأرقام) مثال: {RANDOM:8} → 12345678
• {SEQUENCE} - رقم تسلسلي (6 أرقام) مثال: 000001

أمثلة:
• ORD-{RANDOM:8} → ORD-12345678
• ORD-{YEAR}{MONTH}{DAY}-{RANDOM:6} → ORD-20240220-123456
• ORD-{YEAR}-{SEQUENCE} → ORD-2024-000001
                            ')
                            ->columnSpanFull(),
                    ]),
            ]);
    }

    public function content(Schema $schema): Schema
    {
        return $schema
            ->components([
                Form::make([EmbeddedSchema::make('form')])
                    ->id('form')
                    ->livewireSubmitHandler('save')
                    ->footer([
                        Actions::make($this->getFormActions())
                            ->alignment('end')
                            ->key('form-actions'),
                    ]),
            ]);
    }

    protected function getFormActions(): array
    {
        return [
            \Filament\Actions\Action::make('save')
                ->label('حفظ الإعدادات')
                ->submit('save')
                ->color('success'),
        ];
    }

    public function save(): void
    {
        $settings = app(GeneralSettings::class);

        // حفظ إعدادات التحويل والتطبيق
        $settings->points_to_riyal = $this->points_to_riyal;
        $settings->apps_fee = $this->apps_fee ?? 0;
        $settings->site_name = $this->site_name;
        $settings->site_description = $this->site_description;
        $settings->site_email = $this->site_email;
        $settings->site_phone = $this->site_phone;
        $settings->site_address = $this->site_address ?? null;
        $settings->points_expiry_days = $this->points_expiry_days ?? 365;
        $settings->order_number_format = $this->order_number_format ?? 'ORD-{RANDOM:8}';
        $settings->save();
        
        // إذا كان apps_fee null، قم بتعيينه إلى 0
        if ($settings->apps_fee === null) {
            $settings->apps_fee = 0;
            $settings->save();
        }

        Notification::make()
            ->title('تم حفظ الإعدادات بنجاح')
            ->success()
            ->send();
    }

}
