<?php

namespace App\Filament\Pages;

use App\Models\Client;
use App\Services\FcmService;
use Filament\Forms\Components\CheckboxList;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Notifications\Notification as FilamentNotification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Actions;
use Filament\Schemas\Components\EmbeddedSchema;
use Filament\Schemas\Components\Form;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Concerns\InteractsWithSchemas;
use Filament\Schemas\Contracts\HasSchemas;
use Filament\Schemas\Schema;
use Illuminate\Support\Facades\Log;

class SendNotification extends Page implements HasSchemas
{
    use InteractsWithSchemas;

    protected static string | \BackedEnum | null $navigationIcon = 'heroicon-o-bell';

    protected static ?string $title = 'إرسال إشعارات';

    protected static ?string $navigationLabel = 'إرسال إشعارات';

    protected static string | \UnitEnum | null $navigationGroup = 'إدارة العملاء';

    protected static ?int $navigationSort = 2;

    // Form data properties
    public ?string $notification_title = null;
    public ?string $notification_body = null;
    public ?string $notification_icon = 'heroicon-o-bell';
    public ?string $notification_color = 'success';
    public ?string $action_label = null;
    public ?string $action_url = null;
    public ?string $recipient_type = 'all';
    public array $selected_clients = [];
    public ?string $custom_token = null;
    public bool $save_to_database = false;

    public function mount(): void
    {
        $this->recipient_type = 'all';
        $this->notification_icon = 'heroicon-o-bell';
        $this->notification_color = 'success';
        $this->save_to_database = false;
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('محتوى الإشعار')
                    ->description('أدخل عنوان ورسالة الإشعار')
                    ->icon('heroicon-o-document-text')
                    ->components([
                        TextInput::make('notification_title')
                            ->label('عنوان الإشعار')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('مثال: عرض خاص!')
                            ->helperText('العنوان الذي سيظهر في الإشعار')
                            ->columnSpanFull(),
                        Textarea::make('notification_body')
                            ->label('محتوى الإشعار')
                            ->required()
                            ->rows(4)
                            ->placeholder('أدخل محتوى الإشعار هنا...')
                            ->helperText('المحتوى الكامل للإشعار')
                            ->columnSpanFull(),
                    ]),

                Section::make('مظهر الإشعار')
                    ->description('تخصيص شكل الإشعار')
                    ->icon('heroicon-o-paint-brush')
                    ->collapsible()
                    ->components([
                        Select::make('notification_icon')
                            ->label('أيقونة الإشعار')
                            ->options([
                                'heroicon-o-bell' => '🔔 جرس (إشعار)',
                                'heroicon-o-megaphone' => '📢 مكبر صوت (إعلان)',
                                'heroicon-o-gift' => '🎁 هدية (عرض)',
                                'heroicon-o-tag' => '🏷️ تسمية (خصم)',
                                'heroicon-o-wrench-screwdriver' => '🔧 أدوات (صيانة)',
                                'heroicon-o-check-circle' => '✅ تم (تأكيد)',
                                'heroicon-o-exclamation-triangle' => '⚠️ تحذير',
                                'heroicon-o-information-circle' => 'ℹ️ معلومات',
                                'heroicon-o-star' => '⭐ نجمة (مميز)',
                                'heroicon-o-heart' => '❤️ قلب',
                                'heroicon-o-shopping-cart' => '🛒 طلب',
                                'heroicon-o-calendar' => '📅 تقويم (موعد)',
                                'heroicon-o-currency-dollar' => '💰 مال (دفع)',
                                'heroicon-o-chat-bubble-left' => '💬 رسالة',
                            ])
                            ->native(false)
                            ->default('heroicon-o-bell')
                            ->columnSpan(1),
                        Select::make('notification_color')
                            ->label('لون الإشعار')
                            ->options([
                                'success' => '🟢 أخضر (نجاح)',
                                'info' => '🔵 أزرق (معلومات)',
                                'warning' => '🟡 أصفر (تحذير)',
                                'danger' => '🔴 أحمر (خطر)',
                                'primary' => '🟣 بنفسجي (أساسي)',
                                'gray' => '⚪ رمادي (عادي)',
                            ])
                            ->native(false)
                            ->default('success')
                            ->columnSpan(1),
                    ])->columns(2),

                Section::make('إجراء الإشعار (اختياري)')
                    ->description('أضف زر أو رابط داخل الإشعار')
                    ->icon('heroicon-o-cursor-arrow-ripple')
                    ->collapsible()
                    ->collapsed()
                    ->components([
                        TextInput::make('action_label')
                            ->label('نص الزر')
                            ->placeholder('مثال: عرض التفاصيل')
                            ->helperText('النص الذي سيظهر على الزر داخل الإشعار')
                            ->columnSpan(1),
                        TextInput::make('action_url')
                            ->label('رابط الزر')
                            ->url()
                            ->placeholder('https://example.com/page')
                            ->helperText('الرابط الذي سيفتح عند الضغط على الزر')
                            ->columnSpan(1),
                    ])->columns(2),

                Section::make('المستلمون')
                    ->description('اختر من سيستلم الإشعار')
                    ->icon('heroicon-o-users')
                    ->components([
                        Select::make('recipient_type')
                            ->label('نوع المستلمين')
                            ->options([
                                'all' => 'جميع العملاء النشطين',
                                'selected' => 'عملاء محددون',
                                'token' => 'إرسال لـ FCM Token محدد',
                            ])
                            ->required()
                            ->live()
                            ->native(false)
                            ->columnSpanFull(),

                        CheckboxList::make('selected_clients')
                            ->label('اختر العملاء')
                            ->options(
                                Client::where('is_active', true)
                                    ->get()
                                    ->mapWithKeys(fn ($client) => [
                                        $client->id => $client->name . ' (' . $client->email . ')' . ($client->fcm_token ? ' ✓' : ' ⚠️ بدون Token')
                                    ])
                                    ->toArray()
                            )
                            ->visible(fn () => $this->recipient_type === 'selected')
                            ->searchable()
                            ->helperText('العملاء بدون FCM Token سيتم تخطيهم تلقائياً عند الإرسال')
                            ->columnSpanFull(),

                        TextInput::make('custom_token')
                            ->label('FCM Token')
                            ->placeholder('أدخل FCM Token هنا...')
                            ->helperText('أدخل FCM Token مباشرة لإرسال الإشعار')
                            ->visible(fn () => $this->recipient_type === 'token')
                            ->columnSpanFull(),
                    ]),
            ]);
    }

    public function content(Schema $schema): Schema
    {
        return $schema
            ->components([
                Form::make([EmbeddedSchema::make('form')])
                    ->id('form')
                    ->livewireSubmitHandler('send')
                    ->footer([
                        Actions::make($this->getFormActions())
                            ->alignment('end')
                            ->key('form-actions'),
                    ]),
            ]);
    }

    protected function getFormActions(): array
    {
        return [
            \Filament\Actions\Action::make('send')
                ->label('إرسال الإشعار')
                ->submit('send')
                ->color('success')
                ->icon('heroicon-o-paper-airplane'),
        ];
    }

    /**
     * بناء بيانات الإشعار الكاملة
     */
    protected function buildNotificationPayload(): array
    {
        $data = [];

        // إضافة بيانات المظهر
        $data['icon'] = $this->notification_icon ?? 'heroicon-o-bell';
        $data['color'] = $this->notification_color ?? 'success';

        // إضافة بيانات الإجراء
        if (!empty($this->action_label) && !empty($this->action_url)) {
            $data['action_label'] = $this->action_label;
            $data['action_url'] = $this->action_url;
        }

        return $data;
    }

    public function send(): void
    {
        $fcmService = app(FcmService::class);
        $results = [
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
        ];

        $payload = $this->buildNotificationPayload();

        try {
            switch ($this->recipient_type) {
                case 'all':
                    $results = $fcmService->sendToAllClients(
                        $this->notification_title,
                        $this->notification_body,
                        $payload
                    );
                    break;

                case 'selected':
                    if (empty($this->selected_clients)) {
                        FilamentNotification::make()
                            ->title('خطأ')
                            ->body('يرجى اختيار عملاء على الأقل')
                            ->danger()
                            ->send();
                        return;
                    }
                    $clients = Client::whereIn('id', $this->selected_clients)
                        ->whereNotNull('fcm_token')
                        ->get();
                    $skipped = count($this->selected_clients) - $clients->count();
                    $results = $fcmService->sendToClients($clients, $this->notification_title, $this->notification_body, $payload);
                    $results['skipped'] += $skipped;
                    break;

                case 'token':
                    if (empty($this->custom_token)) {
                        FilamentNotification::make()
                            ->title('خطأ')
                            ->body('يرجى إدخال FCM Token')
                            ->danger()
                            ->send();
                        return;
                    }
                    $success = $fcmService->sendToToken(
                        $this->custom_token,
                        $this->notification_title,
                        $this->notification_body,
                        $payload
                    );
                    $results = [
                        'success' => $success ? 1 : 0,
                        'failed' => $success ? 0 : 1,
                        'skipped' => 0,
                    ];
                    break;
            }

            // إظهار رسالة النجاح
            $message = "تم إرسال الإشعار بنجاح!\n";
            $message .= "نجح: {$results['success']}\n";
            if ($results['failed'] > 0) {
                $message .= "فشل: {$results['failed']}\n";
            }
            if ($results['skipped'] > 0) {
                $message .= "تم تخطي: {$results['skipped']} (لا يوجد FCM Token)";
            }

            FilamentNotification::make()
                ->title('تم إرسال الإشعار')
                ->body($message)
                ->success()
                ->send();

            // إعادة تعيين النموذج
            $this->notification_title = null;
            $this->notification_body = null;
            $this->notification_icon = 'heroicon-o-bell';
            $this->notification_color = 'success';
            $this->action_label = null;
            $this->action_url = null;
            $this->selected_clients = [];
            $this->custom_token = null;
            $this->recipient_type = 'all';

        } catch (\Exception $e) {
            Log::error('Failed to send FCM notification: ' . $e->getMessage());

            FilamentNotification::make()
                ->title('خطأ في إرسال الإشعار')
                ->body('حدث خطأ أثناء إرسال الإشعار: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }
}
