<?php

namespace App\Exports;

use App\Models\Subscription;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class SubscriptionsExport implements FromCollection, WithHeadings, WithMapping, WithStyles
{
    protected $subscriptions;

    public function __construct($subscriptions)
    {
        $this->subscriptions = $subscriptions;
    }

    public function collection()
    {
        return $this->subscriptions;
    }

    public function headings(): array
    {
        return [
            'اسم العميل',
            'الباقة',
            'نوع الاشتراك',
            'تاريخ البداية',
            'تاريخ الانتهاء',
            'الأيام المتبقية',
            'الحالة',
            'المبلغ المدفوع',
            'طريقة الدفع',
            'حالة الدفع',
            'أنواع العقارات',
            'المرافق الإضافية',
        ];
    }

    public function map($subscription): array
    {
        $package = $subscription->package;
        $packageName = $package ? ($package->getTranslation('name', 'ar') . ' / ' . $package->getTranslation('name', 'en')) : '-';
        
        $durationType = $package && $package->duration_days
            ? ($package->duration_days == 365 ? 'سنوي' : 'نصف سنوي')
            : '-';

        $remainingDays = '-';
        if ($subscription->status === 'active' && $subscription->end_date) {
            $days = now()->startOfDay()->diffInDays($subscription->end_date, false);
            $remainingDays = $days >= 0 ? $days . ' يوم' : 'منتهي';
        }

        $statusLabels = [
            'active' => 'نشط',
            'expired' => 'منتهي',
            'cancelled' => 'ملغي',
            'suspended' => 'معلق',
        ];

        $paymentMethodLabels = [
            'cash' => 'كاش',
            'visa' => 'فيزا',
            'mastercard' => 'ماستركارد',
            'mada' => 'مدى',
            'transfer' => 'تحويل بنكي',
            'other' => 'أخرى',
        ];

        $paymentStatusLabels = [
            'paid' => 'مدفوع',
            'pending' => 'معلق',
            'refunded' => 'مسترد',
        ];

        $propertyTypes = $subscription->propertyTypes->map(function ($pt) {
            return $pt->getTranslation('name', 'ar') . ' (' . $pt->pivot->quantity . ')';
        })->implode(', ') ?: '-';

        $facilities = $subscription->facilities->map(function ($f) {
            return $f->getTranslation('name', 'ar') . ' (' . $f->pivot->quantity . ')';
        })->implode(', ') ?: '-';

        return [
            $subscription->client->name ?? '-',
            $packageName,
            $durationType,
            $subscription->start_date->format('Y-m-d'),
            $subscription->end_date->format('Y-m-d'),
            $remainingDays,
            $statusLabels[$subscription->status] ?? $subscription->status,
            number_format($subscription->amount_paid, 2) . ' ر.س',
            $paymentMethodLabels[$subscription->payment_method] ?? ($subscription->payment_method ?? '-'),
            $paymentStatusLabels[$subscription->payment_status] ?? $subscription->payment_status,
            $propertyTypes,
            $facilities,
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true, 'size' => 12]],
        ];
    }
}



